package nestedstate;

import java.util.*;

/**
* This is the context of a FSM. A context holds a reference to the current state
* and also functions as a repository for objects. To do that it extends from
* java.util.Hashtable. Objects can be stored with the hashtable's put method and
* retrieved with the hashtables get method.
*/
public class FSMContext extends Hashtable
{
        private NodeState rootState = null;
        private State firstState;
        private FSMAction initialAction = (FSMAction)LeafState.skip;
                
        
        /**
        * Create a new context with s as the first state
        * @param s The first state
        */
        public FSMContext(NodeState root)
        {
                super();
                setRoot(root);
                getRoot().setState( (getRoot().getDefaultState()) );
                initialize();
        }
        
        /**
        * Create a new context with s as the first state. an then execute i to
        * initialize the context.
        * @param s The first state
        * @param i The initial action
        */
        public FSMContext(NodeState root, FSMAction i)
        {
                super();
                setRoot(root);
                getRoot().setState(getRoot().getDefaultState());
                setInitialAction(i);
                initialize();
        }
        
        public void setRoot(NodeState r) { rootState = r; }
        public NodeState getRoot() { return rootState; }
        
        /**
        * Find out what events can be sent to the current state
        * @return A list of events
        */
        public Vector getEvents()
        {
                return getRoot().getState().getEvents();
        }
               
        
        /**
        * Initialize the context. Clear the hashtable. Set first state and 
        * execute initial action.
        */
        public void initialize()
        {
                this.clear(); // might cause problems if you try to configure
                              // context manually
                initialAction.execute(this, null);
                LeafState l = NodeState.getLowestDefaultLeaf(getRoot());
                l.getStateEntryAction().execute(this, null);
                l.getParent().setState(l); // set the first state
        }
        
                
        public void setInitialAction(FSMAction a) { initialAction = a; }
        public FSMAction getInitialAction() { return initialAction; }
        
        /**
        * Dispatch an event e.
        * @param e The event
        - @param data Some additional data
        */
        public void dispatch(FSMEvent e, Object data)
        {
                getRoot().getState().dispatch(e, data, this);
        }
        
        public State getState()
        {
                return getRoot().getState();
        }
}